/**
 *
 * Copyright 2012-2015 Gears for Breakfast ApS. All Rights Reserved.
 * Applies a slowing freeze, every 2 seconds the player takes damage as long as they stay inside.
 */

class SS_FreezingFogVolume extends Volume
	placeable;

var() Name VolumeName;
var(Volume) Array<int> PermittedActs;
var(Volume) Array<int> ForbiddenActs;
var transient bool HasCheckedPermissions;
var(Volume) bool Enabled;

defaultproperties
{
    bStatic = false;
	Enabled = true;
}

function CheckPermissions()
{
	HasCheckedPermissions = true;
	
	if (Role == Role_Authority && Owner == None && CreationTime <= 0 && Enabled)
	{
		if (PermittedActs.Length > 0 && PermittedActs.Find(Hat_GameManager(worldinfo.game).GetCurrentAct()) == INDEX_NONE)
		{
			Enabled = false;
			return;
		}
		if (ForbiddenActs.Length > 0 && ForbiddenActs.Find(Hat_GameManager(worldinfo.game).GetCurrentAct()) != INDEX_NONE)
		{
			Enabled = false;
			return;
		}
	}
}

simulated function bool CanActivate(Actor Other)
{
	if (!HasCheckedPermissions)
		CheckPermissions();
	
	if (!Enabled) return false;
	if (VolumeName == '') return false;
	if (Hat_Player(Other) != None && !Hat_Player(Other).bIsPlayerOwner) return false;
	if (!Other.IsA('Pawn')) return false;
	if (Pawn(Other).Controller == None) return false;
	if (!Pawn(Other).Controller.IsA('Hat_PlayerController')) return false;
	
	return true;
}

simulated function bool IsTouchingOtherRelevantMusicVolumes(Actor Other)
{
	local SS_FreezingFogVolume Fog;
	foreach Hat_Player(Other).TouchingActors(class'SS_FreezingFogVolume', Fog)
	{
		if (Fog == self) continue;
		if (!Fog.CanActivate(Other)) continue;
		return true;
	}
	return false;
}

simulated event Touch( Actor Other, PrimitiveComponent OtherComp, vector HitLocation, vector HitNormal )
{
	local SS_StatusEffect_FreezingFog freezingfog;
	local SS_StatusEffect_IcePower ice;
	

	if(!Other.IsA('Hat_Player')) return;
	
	
	
	freezingfog = SS_StatusEffect_FreezingFog(Hat_Player(Other).GetStatusEffect(class'SS_StatusEffect_FreezingFog',false));
	
	
	if (!HasCheckedPermissions)
		CheckPermissions();
	
	if (CanActivate(Other))
	{
		`SetMusicParameterInt(VolumeName, 1);
		if(Hat_Player(Other).HasStatusEffect(class'SS_StatusEffect_FreezingFog'))
			{
				freezingfog.LeftVolume = false;
				freezingfog.VolumeExitTimer = 0.75;
			}
		else
			Hat_Player(Other).GiveStatusEffect(class'SS_StatusEffect_FreezingFog');
			
		if(Hat_Player(Other).HasStatusEffect(class'SS_StatusEffect_IcePower', false) && CheckForFogUpgrade(Other))
		{
			ice = SS_StatusEffect_IcePower(Hat_Player(Other).GetStatusEffect(class'SS_StatusEffect_IcePower',false));
			ice.InsideFog = true; // Causes the ability to last longer while inside the fog
		}	
	}

    Super.Touch( Other, OtherComp, HitLocation, HitNormal );
}

function bool CheckForFogUpgrade(Actor Other)
{
	local Hat_PlayerController pc;
	
	pc = Hat_PlayerController(Pawn(Other).Controller);
	if (pc.GetLoadout().HasCollectible(class'SS_Collectible_Shard_Ice', 1))
		return true;
	return false;
}

simulated event UnTouch( Actor Other )
{
	local SS_StatusEffect_FreezingFog freezingfog;
	local SS_StatusEffect_IcePower ice;
	
	if(!Other.IsA('Hat_Player')) return;
	
	freezingfog = SS_StatusEffect_FreezingFog(Hat_Player(Other).GetStatusEffect(class'SS_StatusEffect_FreezingFog',false));
	
	if (Hat_Player(Other) != None && !Hat_Player(Other).bIsPlayerOwner) return;
	
	if (CanActivate(Other) && !IsTouchingOtherRelevantMusicVolumes(Other))
	{
		`SetMusicParameterInt(VolumeName, 0);

		if(Hat_Player(Other).HasStatusEffect(class'SS_StatusEffect_FreezingFog'))
		{
			freezingfog.LeftVolume = true;
		}
		
		if(Hat_Player(Other).HasStatusEffect(class'SS_StatusEffect_IcePower', false) && CheckForFogUpgrade(Other))
		{
			ice = SS_StatusEffect_IcePower(Hat_Player(Other).GetStatusEffect(class'SS_StatusEffect_IcePower',false));
			ice.InsideFog = false;
		}	
	}

    Super.UnTouch(Other);
}

simulated function Deactivate()
{
	local Actor a;

	SetCollision(false, bBlockActors);
	Enabled = false;

	foreach TouchingActors(class'Actor', a)
	{
		UnTouch(a);
	}
}